import pandas as pd
import time
import shutil
from pathlib import Path
from jinja2 import Environment, FileSystemLoader, select_autoescape
from datetime import datetime
import multiprocessing
import os
import sys # Pievienots
import traceback
from functools import partial
import random

# ==============================================================================
# IZMAIŅAS: CEĻU IESTATĪŠANA MODUĻU IMPORTAM
# ==============================================================================
# Nosakām skripta atrašanās vietu (tagad tas ir 'python_generator' mapē)
SCRIPT_DIR = Path(__file__).parent.resolve()

# Pievienojam 'kods' apakšmapi pie sistēmas ceļiem, lai Python varētu atrast
# moduļus (config, processing, core, templates), kas atrodas tur.
sys.path.append(str(SCRIPT_DIR / 'kods'))

# ==============================================================================
# IMPORTI NO 'kods' MAPES (darbojas, pateicoties sys.path.append)
# ==============================================================================
from config import settings
from processing import loader, financial_engine, formatters, segmenter
from core import data_fetcher, page_builder
from templates.TXT.descriptions import descriptions, financial_descriptions, DESCRIPTION_DISCLAIMER

# ==============================================================================
# Paralēlās apstrādes palīgfunkcijas
# ==============================================================================

worker_data = {}

def init_worker(dataframes_main, templates_dir_main, company_pages_base_dir_main, nace_df_main):
    """
    Inicializācijas funkcija katram darba procesam.
    """
    global worker_data
    worker_data['dataframes'] = dataframes_main
    worker_data['templates_dir'] = templates_dir_main
    worker_data['company_pages_base_dir'] = company_pages_base_dir_main
    worker_data['nace_df'] = nace_df_main

    env = Environment(loader=FileSystemLoader(templates_dir_main), autoescape=select_autoescape(['html', 'xml', 'php']), trim_blocks=True, lstrip_blocks=True)
    env.filters['format_number'] = lambda val, cur='EUR', factor=1: formatters.format_number_data(val, cur, factor)
    env.filters['format_plain_number'] = lambda val: f"{int(val):,}".replace(",", " ") if val is not None and isinstance(val, (int, float)) and val == int(val) else (f"{val:,.0f}".replace(",", " ") if val is not None and isinstance(val, (int, float)) else ("—" if val is None else val))
    worker_data['env'] = env

def generate_company_description(registered_date_str):
    """
    Ģenerē tekstuālu aprakstu par uzņēmumu, balstoties uz tā vecumu.
    """
    if not registered_date_str or not isinstance(registered_date_str, str):
        return ""

    try:
        # Explicitly specify the date format to avoid ambiguity and warnings.
        # errors='coerce' will turn unparseable dates into NaT (Not a Time), which is handled by the except block.
        registration_date = pd.to_datetime(registered_date_str, format='%Y-%m-%d', errors='coerce').to_pydatetime()
        current_year = datetime.now().year
        age = current_year - registration_date.year
    except (ValueError, IndexError):
        return ""

    for (min_age, max_age), phrases in descriptions.items():
        if min_age <= age <= max_age:
            return random.choice(phrases)

    return ""

def get_ratio_sentence(ratio_value, thresholds, sentences):
    """
    Ģenerē teikumu finanšu rādītājam, balstoties uz vērtību un sliekšņiem, un pievieno krāsu.
    """
    if ratio_value is None:
        return None

    # Vispārējā krāsu karte visiem rādītāju līmeņiem
    color_map = {
        # Negatīvs (Sarkans)
        'danger': 'red',
        'distress': 'red',
        'high_risk': 'red',
        'low': 'red',

        # Piesardzība (Oranžs)
        'precaution': 'orange',
        'grey': 'orange',
        'elevated_risk': 'orange',
        'borderline': 'orange',
        'satisfactory': 'orange',
        'average': 'orange',

        # Pozitīvs (Zaļš)
        'healthy': 'green',
        'safe': 'green',
        'low_risk': 'green',
        'high': 'green',

        # Īpašais (Zils)
        'excessive': 'blue',
        'inefficient': 'blue'
    }

    category = None
    # Īpašs gadījums Altman Z-Score
    if 'distress' in thresholds:
        if ratio_value < thresholds['distress']:
            category = 'distress'
        elif ratio_value <= thresholds['grey']:
            category = 'grey'
        else:
            category = 'safe'
    # Jauna loģika 4 līmeņu rādītājiem
    elif 'four_level' in thresholds:
        keys = list(sentences.keys())
        if ratio_value < thresholds['level1_max']:
            category = keys[0]
        elif ratio_value < thresholds['level2_max']:
            category = keys[1]
        elif ratio_value < thresholds['level3_max']:
            category = keys[2]
        else:
            category = keys[3]
    # Īpašs gadījums Debt-to-Equity, kur zemāks ir labāks
    elif thresholds.get('invert_logic'):
        keys = list(sentences.keys())
        if ratio_value < thresholds['low']:
            category = keys[0]
        elif ratio_value <= thresholds['high']:
            category = keys[1]
        else:
            category = keys[2]
    # Standarta 3 līmeņu gadījums
    else:
        keys = list(sentences.keys())
        if ratio_value < thresholds['low']:
            category = keys[0]
        elif ratio_value <= thresholds['high']:
            category = keys[1]
        else:
            category = keys[2]

    if category and category in sentences:
        sentence = random.choice(sentences[category])
        if category in color_map:
            color = color_map[category]
            sentence = sentence.replace('<br>', '')
            return f'<span style="color: {color};">{sentence}</span><br>'
        return sentence  # Atgriež oriģinālo teikumu, ja nav krāsu kartes
    return None


def process_company(reg_nr):
    """
    Funkcija, kas apstrādā VIENU uzņēmumu.
    """
    dataframes = worker_data['dataframes']
    env = worker_data['env']
    company_pages_base_dir = worker_data['company_pages_base_dir']
    nace_df = worker_data['nace_df']

    try:
        table_names_to_process = list(dataframes.keys())
        company_main_data = data_fetcher.fetch_main_company_data(dataframes.get('register'), reg_nr)
        all_company_results = data_fetcher.fetch_all_data_for_reg_nr(dataframes, reg_nr, table_names_to_process)
        member_as_entity = data_fetcher.fetch_member_as_entity_data(dataframes.get('members'), reg_nr)
        fs_info, fs_ids = financial_engine.get_financial_statements_info(dataframes.get('financial_statements'), reg_nr)
        income_raw, balance_raw, cash_flow_raw = financial_engine.get_financial_details_data(dataframes, fs_ids)
        all_processed_financials, sankey_years = financial_engine.process_financial_data_for_years(fs_info, income_raw, balance_raw, cash_flow_raw)

        summary_table_data = page_builder.prepare_summary_table_data(all_processed_financials, sankey_years)
        segment_info = segmenter.determine_company_segment(company_main_data, summary_table_data)
        
        # === APRAKSTA ĢENERĒŠANA (FINĀLA VERSIJA) ===
        description_sentences = []

        # 1. Teikums par vecumu, sākot ar uzņēmuma nosaukumu
        registered_date = company_main_data.get('registered')
        age_description = generate_company_description(registered_date)
        if age_description:
            # Convert to string before stripping to avoid errors with non-string (e.g., float/NaN) values.
            company_type = str(company_main_data.get('type', '')).strip()
            name_in_quotes = str(company_main_data.get('name_in_quotes', '')).strip()

            # Veidojam nosaukuma daļu tikai, ja abi lauki ir pieejami
            if company_type and name_in_quotes:
                # Izveidojam ievaddaļu ar uzņēmuma nosaukumu
                company_name_intro = f'{company_type} "{name_in_quotes}"'
                
                # Saliekam teikumu pēc lietotāja norādītās struktūras: "[Nosaukums] ir [Oriģinālais teikums]"
                full_first_sentence = f"{company_name_intro} ir {age_description}"
                description_sentences.append(full_first_sentence)
            else:
                # Ja nosaukuma daļas trūkst, izmantojam oriģinālo teikumu
                description_sentences.append(age_description)

        # 2. Finanšu teikumi - izmantojam datus no `financial_engine`
        try:
            # Atrodam pēdējā gada finanšu datus
            latest_year_data = None
            if sankey_years:
                latest_year = max(sankey_years)

                # Pievienojam teikumu ar jaunāko gadu
                year_sentence = f"Pamatojoties uz {latest_year}. gada datiem, "
                description_sentences.append(year_sentence)

                if all_processed_financials.get(latest_year):
                    # Dodam priekšroku UGP (Uzņēmuma gada pārskats), ja pieejams
                    latest_year_data = all_processed_financials[latest_year].get('UGP') or all_processed_financials[latest_year].get('UKGP')

            if latest_year_data and latest_year_data.get('financial_ratios'):
                calculated_ratios = latest_year_data['financial_ratios']

                # Atjaunotās definīcijas, kas atbilst jaunajai get_ratio_sentence loģikai un descriptions.py failam
                ratio_definitions = {
                    'current_ratio': {
                        'thresholds': {'four_level': True, 'level1_max': 1.0, 'level2_max': 1.5, 'level3_max': 3.0}
                    },
                    'quick_ratio': {
                        'thresholds': {'four_level': True, 'level1_max': 0.8, 'level2_max': 1.0, 'level3_max': 2.0}
                    },
                    'net_profit_margin': {
                        'thresholds': {'low': 0, 'high': 10}
                    },
                    'roa': {
                        'thresholds': {'low': 2, 'high': 5}
                    },
                    'roe': {
                        'thresholds': {'low': 5, 'high': 15}
                    },
                    'debt_to_equity': {
                        'thresholds': {'low': 1.5, 'high': 2.0, 'invert_logic': True}
                    },
                    'asset_turnover': {
                        'thresholds': {'low': 0.5, 'high': 1.0}
                    },
                    'roce': {
                        'thresholds': {'low': 5, 'high': 15}
                    },
                    'altman_z_score': {
                        'thresholds': {'distress': 1.8, 'grey': 2.99}
                    }
                }

                for name, definition in ratio_definitions.items():
                    ratio_data = calculated_ratios.get(name)
                    if ratio_data:
                        ratio_value = ratio_data.get('value')
                        
                        # Pārveidojam procentuālās vērtības, ja nepieciešams
                        if name in ['net_profit_margin', 'roa', 'roe', 'roce']:
                             if ratio_value is not None:
                                ratio_value *= 100

                        sentences = financial_descriptions.get(name)
                        if sentences:
                            sentence = get_ratio_sentence(ratio_value, definition['thresholds'], sentences)
                            if sentence:
                                description_sentences.append(sentence)
        except Exception as e:
            # Klusa kļūda, lai neietekmētu galveno procesu, ja aprēķins neizdodas
            print(f"[DEBUG] Kļūda finanšu teikumu ģenerēšanā: {e}")
            pass

        full_description = " ".join(description_sentences)
        full_description += f"<br><br>{DESCRIPTION_DISCLAIMER}"

        page_data = {
            'search_reg_nr': reg_nr,
            'dati_php_rowData': company_main_data.to_dict() if company_main_data is not None else {},
            'results': all_company_results,
            'member_as_entity_data': member_as_entity,
            'allProcessedData': all_processed_financials,
            'sankeyAvailableYears': sankey_years,
            'errors': [],
            'segment': segment_info,
            'summary_table_data_for_js': summary_table_data,
            'company_description': full_description
        }
        
        # === SĀKUMS: MODIFICĒTA APSTRĀDES SECĪBA ===
        # 1. Sagatavojam visus datu blokus
        panel_details = page_builder.get_company_details_for_panel(company_main_data, reg_nr, segment_info)
        nace_info = page_builder.get_company_nace_info(all_company_results, nace_df)
        vid_panel_data = page_builder.prepare_vid_panel_data(all_company_results)

        # 2. Pievienojam visus sagatavotos datus galvenajam datu objektam
        panel_details.update(nace_info)
        page_data.update(panel_details)
        page_data['vid_panel_data'] = vid_panel_data

        # 3. Tikai tagad, kad visi dati ir pieejami, ģenerējam SEO/FAQ sadaļu
        page_data['seo_metadata'] = page_builder.prepare_seo_metadata(page_data)
        # === BEIGAS: MODIFICĒTA APSTRĀDES SECĪBA ===
        
        page_data['search_term_value'] = panel_details['companyTitleForHtml']
        page_data['current_page_title'] = f"{panel_details['companyTitleForHtml']} ({reg_nr})"
        page_data['processed_tables_for_display'] = page_builder.prepare_data_for_results_tables(page_data)
        
        hidden_links_regcodes = set()
        main_super_id = data_fetcher.fetch_super_id(dataframes.get('register'), reg_nr)
        if main_super_id:
            target_ids = [main_super_id + offset for offset in [1, -1, 200, -200] if main_super_id + offset > 0]
            hidden_links_regcodes.update(data_fetcher.fetch_regcodes_by_super_ids(dataframes.get('register'), target_ids, active_only=True))
        
        hidden_links_regcodes.update(data_fetcher.fetch_random_regcodes(dataframes.get('register'), 5, active_only=True))
        
        page_data['has_summary_data'] = bool(summary_table_data.get('UGP') or summary_table_data.get('UKGP'))
        page_data['has_financial_charts'] = bool(sankey_years)

        page_data['generated_hidden_links'] = [f"{settings.BASE_DOMAIN}/{rc}" for rc in sorted(hidden_links_regcodes) if rc != reg_nr]
        page_data['canonical_url'] = f"{settings.BASE_DOMAIN}/{reg_nr}"
        page_data['js_config_json'] = page_builder.build_js_config(page_data)
        page_data['generationDate'] = datetime.now().strftime('%d.%m.%Y')
        page_data['base_project_url'] = settings.BASE_PROJECT_URL_FOR_HTML_LINKS
        page_data['assets_base_path'] = settings.ASSETS_BASE_PATH_FOR_HTML

        template = env.get_template('layout.html')
        html_content = template.render(page_data=page_data)
        
        dir1, dir2 = reg_nr[0:2], reg_nr[2:4]
        target_dir = company_pages_base_dir / dir1 / dir2
        target_dir.mkdir(parents=True, exist_ok=True)
        output_file = target_dir / f"{reg_nr}.php"
        with open(output_file, 'w', encoding='utf-8') as f:
            f.write(html_content)
        
        return reg_nr
    except Exception as e:
        print(f"\n!!! Kļūda, apstrādājot {reg_nr}: {e} !!!\n")
        print("--- PILNA KĻŪDAS IZDRUKA ---")
        traceback.print_exc()
        print("----------------------------\n")
        return None

# ==============================================================================
# SITEMAP UN ROBOTS.TXT ĢENERĒŠANAS FUNKCIJA
# ==============================================================================

def generate_sitemaps_and_robots_txt(reg_nrs, out_dir, base_domain, sitemap_url_limit=500):
    """
    Ģenerē sitemap.xml failus (sadalītus pa daļām) un robots.txt failu.
    """
    sitemaps_dir = out_dir / "sitemaps"
    sitemaps_dir.mkdir(exist_ok=True)

    last_mod_date = datetime.now().strftime('%Y-%m-%d')
    sitemap_files = []

    reg_nr_chunks = [reg_nrs[i:i + sitemap_url_limit] for i in range(0, len(reg_nrs), sitemap_url_limit)]

    for i, chunk in enumerate(reg_nr_chunks):
        sitemap_filename = f"sitemap-{i+1}.xml"
        sitemap_path = sitemaps_dir / sitemap_filename
        
        xml_parts = [
            '<?xml version="1.0" encoding="UTF-8"?>',
            '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">'
        ]
        for reg_nr in chunk:
            xml_parts.append('  <url>')
            xml_parts.append(f'    <loc>{base_domain}/{reg_nr}</loc>')
            xml_parts.append(f'    <lastmod>{last_mod_date}</lastmod>')
            xml_parts.append('  </url>')
        xml_parts.append('</urlset>')
        
        xml_content = "\n".join(xml_parts)
        
        with open(sitemap_path, 'w', encoding='utf-8') as f:
            f.write(xml_content)
        
        sitemap_files.append(sitemap_filename)
        print(f"Izveidots {sitemap_filename} ar {len(chunk)} saitēm.")

    sitemap_index_path = sitemaps_dir / "sitemap.xml"
    
    sitemap_index_parts = [
        '<?xml version="1.0" encoding="UTF-8"?>',
        '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">'
    ]
    for sitemap_file in sitemap_files:
        sitemap_index_parts.append('  <sitemap>')
        sitemap_index_parts.append(f'    <loc>{base_domain}/sitemaps/{sitemap_file}</loc>')
        sitemap_index_parts.append(f'    <lastmod>{last_mod_date}</lastmod>')
        sitemap_index_parts.append('  </sitemap>')
    sitemap_index_parts.append('</sitemapindex>')

    xml_index_content = "\n".join(sitemap_index_parts)
    with open(sitemap_index_path, 'w', encoding='utf-8') as f:
        f.write(xml_index_content)
    print("Izveidots sitemap.xml (indeksa fails).")
    
    robots_parts = [
        "User-agent: *",
        "Allow: /",
        ""
    ]
    for sitemap_file in sitemap_files:
        robots_parts.append(f"Sitemap: {base_domain}/sitemaps/{sitemap_file}")
    robots_parts.append(f"Sitemap: {base_domain}/sitemaps/sitemap.xml")
    
    robots_content = "\n".join(robots_parts)
    with open(out_dir / "robots.txt", 'w', encoding='utf-8') as f:
        f.write(robots_content)
    print("Izveidots robots.txt.")


# ==============================================================================
# GALVENĀ FUNKCIJA
# ==============================================================================
def main():
    start_time_total = time.time()

    # --- 1. Ceļu un vides iestatīšana (IZMAIŅAS JAUNAJAI STRUKTŪRAI) ---
    script_dir = Path(__file__).parent.resolve() # python_generator root
    base_dir = script_dir # Pieņemam, ka csv un out ir blakus šim failam
    kods_dir = script_dir / 'kods' # Mape, kurā palika resursi

    # Ceļi uz datiem un izvadi
    csv_dir = base_dir / "csv"
    out_dir = base_dir / "out"
    
    # Resursi tagad jāmeklē iekš 'kods_dir'
    templates_dir = kods_dir / "templates"
    partials_src_dir = templates_dir / "partials"
    assets_source_dir = templates_dir / "assets"
    php_backend_src_dir = templates_dir / "php_backend"
    
    assets_target_dir = out_dir / "assets"
    search_assets_dest_dir = assets_target_dir / "search"
    company_pages_base_dir = out_dir / "x"

    # Pozīcija avots tagad ir iekš 'kods'
    pozicija_source_dir = kods_dir / "pozicija"
    pozicija_target_dir = out_dir / "pozicija"

    # SQLite datubāze arī ir iekš 'kods'
    sqlite_source_path = kods_dir / "companies.sqlite"
    sqlite_dest_path = search_assets_dest_dir / "companies.sqlite"

    # Pārliecināmies, ka izvades mape eksistē
    out_dir.mkdir(exist_ok=True)
    
    # Kopējam 'assets' no kods/templates/assets uz out/assets
    if assets_target_dir.exists(): 
        shutil.rmtree(assets_target_dir)
    if assets_source_dir.exists(): 
        shutil.copytree(assets_source_dir, assets_target_dir)
        print(f"Direktorija '{assets_source_dir}' veiksmīgi nokopēta uz '{assets_target_dir}'.")
    else:
        print(f"BRĪDINĀJUMS: Avota direktorija '{assets_source_dir}' netika atrasta. 'assets' netiks kopēti.")

    # Kopējam 'pozicija' mapi no kods/pozicija uz out/pozicija
    if pozicija_target_dir.exists():
        shutil.rmtree(pozicija_target_dir)
    if pozicija_source_dir.exists():
        shutil.copytree(pozicija_source_dir, pozicija_target_dir)
        print(f"Direktorija '{pozicija_source_dir}' veiksmīgi nokopēta uz '{pozicija_target_dir}'.")
    else:
        print(f"BRĪDINĀJUMS: Avota direktorija '{pozicija_source_dir}' netika atrasta. 'pozicija' netiks kopēta.")

    # Sagatavojam meklēšanas resursus
    search_assets_dest_dir.mkdir(parents=True, exist_ok=True)
    
    if sqlite_source_path.exists(): 
        shutil.copy2(sqlite_source_path, sqlite_dest_path)
    else:
        sqlite_dest_path.touch() # Lai nebūtu kļūdas, ja DB vēl nav ģenerēta

    src_search_php = php_backend_src_dir / "search.php"
    dest_search_php = search_assets_dest_dir / "search.php"
    if src_search_php.exists(): 
        shutil.copy2(src_search_php, dest_search_php)
        
    src_header_php = partials_src_dir / "header.php"
    dest_header_php = out_dir / "header.php"
    if src_header_php.exists(): 
        shutil.copy2(src_header_php, dest_header_php)
        print(f"Fails '{src_header_php}' veiksmīgi nokopēts uz '{dest_header_php}'.")
    else:
        print(f"BRĪDINĀJUMS: Avota fails '{src_header_php}' netika atrasts. 'header.php' netiks kopēts.")

    # --- 2. Datu ielāde ---
    print("\nIelādē CSV datus...")
    if not csv_dir.is_dir():
        print(f"Kļūda: CSV direktorija nav atrasta: {csv_dir}")
        return

    # Papildus ielādējam NACE klasifikatoru (IZMAIŅA: meklē 'kods' mapē)
    try:
        nace_df = pd.read_csv(kods_dir / 'NACE.csv', dtype={'Kods': str})
        print("NACE klasifikators veiksmīgi ielādēts.")
    except FileNotFoundError:
        print(f"Kļūda: NACE.csv fails nav atrasts direktorijā: {kods_dir}")
        return

    table_names_to_process = [p.stem for p in csv_dir.glob('*.csv')]
    dataframes = loader.load_all_csv_data(csv_dir, table_names_to_process)
    print("Datu ielāde pabeigta.")

    if 'register' not in dataframes or dataframes['register'].empty:
        print("Kļūda: register.csv fails trūkst vai ir tukšs. Darbība tiek pārtraukta.")
        return

    # --- 3. Apstrādājamo reģ. nr. noteikšana ---
    all_valid_reg_nrs = set(dataframes['register']['regcode'].dropna())
    all_valid_reg_nrs = {reg_nr for reg_nr in all_valid_reg_nrs if isinstance(reg_nr, str) and reg_nr.isdigit() and len(reg_nr) == 11}
    
    user_input = input("Ievadiet 11 ciparu reģistrācijas numuru vai 'all', lai ģenerētu visus failus: ").strip().lower()

    reg_nrs_to_process = []
    if user_input == 'all':
        reg_nrs_to_process = sorted(list(all_valid_reg_nrs))
    elif user_input.isdigit() and len(user_input) == 11 and user_input in all_valid_reg_nrs:
        reg_nrs_to_process.append(user_input)
    else:
        print("Nederīga ievade.")
        return

    # --- 4. Paralēlās ģenerēšanas cikls ---
    if not reg_nrs_to_process:
        print("Nav uzņēmumu, ko apstrādāt.")
        return

    total_to_process = len(reg_nrs_to_process)
    print(f"\nSāk ģenerēt PHP failus {total_to_process} uzņēmumiem.")

    try:
        num_processes = os.cpu_count()
        print(f"Sistēmā ir {num_processes} kodoli. Visi tiks izmantoti (100%).")
    except (NotImplementedError, TypeError):
        num_processes = 4
        print(f"Nevarēja noteikt kodolu skaitu. Tiks izmantoti {num_processes} procesi.")

    # Pievienots nace_df kā pēdējais initargs arguments
    with multiprocessing.Pool(processes=num_processes,
                              initializer=init_worker,
                              initargs=(dataframes, templates_dir, company_pages_base_dir, nace_df)) as pool:
        
        processed_count = 0
        for result in pool.imap_unordered(process_company, reg_nrs_to_process):
            if result:
                processed_count += 1
                print(f"Progres: [{processed_count}/{total_to_process}]", end="\r")

    print(f"\nApstrāde pabeigta. Kopā ģenerēti {processed_count} faili.")

    # --- 5. Papildu lapu (index.php, 404.php) ģenerēšana ---
    env = Environment(loader=FileSystemLoader(templates_dir), autoescape=select_autoescape(['html', 'xml', 'php']), trim_blocks=True, lstrip_blocks=True)
    
    try:
        print("Ģenerē sākuma lapu (index.php)...")
        index_template = env.get_template('index.html')
        index_page_data = {'search_term_value': '', 'assets_base_path': settings.ASSETS_BASE_PATH_FOR_HTML }
        index_content = index_template.render(page_data=index_page_data)
        index_output_file = out_dir / "index.php"
        with open(index_output_file, 'w', encoding='utf-8') as f:
            f.write(index_content)
        print("Sākuma lapa veiksmīgi izveidota.")
    except Exception as e:
        print(f"Kļūda, ģenerējot index.php: {e}")
        
    try:
        print("Ģenerē 404 kļūdas lapu (404.php)...")
        template_404 = env.get_template('404.html')
        page_data_404 = {
            'search_term_value': '',
            'assets_base_path': settings.ASSETS_BASE_PATH_FOR_HTML,
            'base_project_url': settings.BASE_PROJECT_URL_FOR_HTML_LINKS,
            'current_page_title': '404 - Lapa nav atrasta'
        }
        content_404 = template_404.render(page_data=page_data_404)
        output_file_404 = out_dir / "404.php"
        with open(output_file_404, 'w', encoding='utf-8') as f:
            f.write(content_404)
        print("404 lapa veiksmīgi izveidota.")
    except Exception as e:
        print(f"Kļūda, ģenerējot 404.php: {e}")
    
    # --- 6. Sitemap un robots.txt ģenerēšana ---
    if reg_nrs_to_process:
        print("\nSāk ģenerēt sitemap un robots.txt failus...")

        # Atlasa tikai aktīvos uzņēmumus priekš sitemap
        register_df = dataframes['register']
        
        # Nosacījumi aktīvam uzņēmumam: 'closed' nav 'L' vai 'R' UN 'terminated' ir tukšs vai '0000-00-00'
        active_mask = (
            (~register_df['closed'].isin(['L', 'R'])) &
            (register_df['terminated'].isna() | register_df['terminated'].isin(['', '0000-00-00']))
        )
        active_reg_nrs_set = set(register_df[active_mask]['regcode'].dropna())

        # Nosaka, kuri no apstrādātajiem uzņēmumiem ir jāiekļauj sitemap
        sitemap_reg_nrs = [nr for nr in reg_nrs_to_process if nr in active_reg_nrs_set]

        if sitemap_reg_nrs:
            print(f"Sitemap tiks ģenerēts {len(sitemap_reg_nrs)} aktīviem uzņēmumiem.")
            generate_sitemaps_and_robots_txt(
                reg_nrs=sitemap_reg_nrs,
                out_dir=out_dir,
                base_domain=settings.BASE_DOMAIN,
                sitemap_url_limit=25000
            )
            print("Sitemap un robots.txt failu ģenerēšana pabeigta.")
        else:
            print("Nav atrasts neviens aktīvs uzņēmums, lai ģenerētu sitemap.")

    total_time_taken = time.time() - start_time_total
    print(f"\nKopējais ģenerēšanas laiks: {total_time_taken:.2f} sekundes.")
    print(f"Izvades direktorija: {out_dir.resolve()}")

if __name__ == "__main__":
    multiprocessing.freeze_support()
    main()